class Renderer {
    canvas
    ctx
    buffers
    palette
    paletteMap
    overscanSize
    outputWidth
    outputHeight
    bufferWidth
    bufferHeight
    origin
    displayPort
  
    constructor (canvas, ctx) {
        this.canvas = canvas
        this.ctx = ctx
        this.setSize(640/2, 480/2, 0)

        this.palette = []
        this.paletteMap = {}

        // Decorate model data..
        Object.entries(models).forEach(([name, model]) => {
            this.decorateModel(model, name)
        })
    }

    setSize (width, height, overscanSize) {
        this.overscanSize = overscanSize
        this.outputWidth = width
        this.outputHeight = height
        this.bufferWidth = width + overscanSize*2
        this.bufferHeight = height + overscanSize*2
        this.canvas.width = width
        this.canvas.height = height

        this.canvasBuffer = ctx.createImageData(this.bufferWidth, this.bufferHeight)
        for (let i = 0; i < this.bufferWidth * this.bufferHeight * 4; ++i) {
            this.canvasBuffer.data[i] = 255
        }
        this.finalImageBuffer = ctx.createImageData(this.outputWidth, this.outputHeight)
        for (let i = 0; i < this.bufferWidth * this.bufferHeight * 4; ++i) {
            this.canvasBuffer.data[i] = 255
        }

        this.buffers = {}
        const createBufferType = (name, count, arrayType) => {
            this.buffers[name] = {
                buffers: Array.from({length: count}, () => { return {
                    data: new arrayType(this.bufferWidth * this.bufferHeight),
                    width: this.bufferWidth,
                    height: this.bufferHeight,
                    pixelCount: this.bufferHeight * this.bufferWidth,
                }}),
                locks: Array.from({length: count}, () => false),
                currentId: null
            }
            this.setCurrentBufferId(name, this.lockBuffer(name))
        }
        createBufferType('color', 2, Uint8Array )
        createBufferType('brightness', 3, Float32Array)
        createBufferType('depth', 2, Float32Array)
    }

    decorateModel (model, name = 'unnamed') {
        const verts = model.verts
        const normals = model.normals = []
        // let badFaces = 0
        // let badNormals = 0
        const storeNormal = (p0, p1, p2) => {
            // if ((p0[0] === p1[0] && p0[1] === p1[1] && p0[2] === p1[2]) ||
            //     (p0[0] === p2[0] && p0[1] === p2[1] && p0[2] === p2[2]) ||
            //     (p1[0] === p2[0] && p1[1] === p2[1] && p1[2] === p2[2])) {
            //     badFaces++
            // }
            // const v01 = m4.subtractVectors(m4.multiplyVector(p0, 10), m4.multiplyVector(p1, 10))
            // const v21 = m4.subtractVectors(m4.multiplyVector(p2, 10), m4.multiplyVector(p1, 10))
            const v01 = m4.subtractVectors(p0, p1)
            const v21 = m4.subtractVectors(p2, p1)
            const normal = m4.normalize(m4.cross(v01, v21))
            // if (normal[0] === 0 && normal[1] === 0 && normal[2] === 0) {
            //     // debugger
            //     badNormals++
            // }
            const found = normals.findIndex(x => x[0]===normal[0] && x[1]===normal[1] && x[2]===normal[2])
            if (found !== -1) {
                return found
            } else {
                normals.push(normal)
                return normals.length - 1
            }
        }
        model.objects.forEach(object => {
            object.subObjects.forEach(subObject => {
                subObject.triNorms = subObject.tris.map(tri => {
                    const p0 = verts[tri[0]]
                    const p1 = verts[tri[1]]
                    const p2 = verts[tri[2]]
                    return storeNormal(p0, p1, p2)
                })
                subObject.quadNorms = subObject.quads.map(quad => {
                    const p0 = verts[quad[0]]
                    const p1 = verts[quad[1]]
                    const p2 = verts[quad[2]]
                    return storeNormal(p0, p1, p2)
                })
            })
        })
        // if (badFaces > 0 || badNormals > 0) {
        //     console.log(`${badFaces} bad faces and ${badNormals} bad normals in ${name}`)
        // }
    }

    getPaletteId (diffuse, emissive) {
        const paletteMapKey = JSON.stringify(diffuse) + JSON.stringify(emissive)
        if (this.paletteMap[paletteMapKey] === undefined) {
            this.paletteMap[paletteMapKey] = this.palette.length
            this.palette.push([
                [emissive[0]*255, emissive[1]*255, emissive[2]*255],
                [diffuse[0]*255, diffuse[1]*255, diffuse[2]*255]
            ])
        }
        return this.paletteMap[paletteMapKey]
    }

    startFrame () {
        this.ctx.fillStyle = '#f0f'
        this.ctx.fillRect(0, 0, this.canvasBuffer.width, this.canvasBuffer.height)
        this.canvas.style.filter = ''

        const pixels = this.canvasBuffer.data
        for (i = 0; i < this.canvasBuffer.width*this.canvasBuffer.height*4; ) {
            pixels[i++] = 255
            pixels[i++] = 0
            pixels[i++] = 255
            i++
        }

        this.origin = [this.bufferWidth / 2, this.bufferHeight / 2]
        this.displayPort = [0, 0, this.bufferWidth-1, this.bufferHeight-1]

        this.palette = [
            [[0, 0, 0], [0, 0, 0]],
        ]
        this.paletteMap = {}

        this.renderStats = {
            badFaces: 0,
            trisConsidered: 0,
            trisBackfaceCulled: 0,
            trisHidden: 0,
            trisRendered: 0,
        }

        this.ambientLightStrength = 1
        this.lights = []

        minz = 9999
        maxz = -9999
    }

    setOrigin (origin) {
        this.origin = origin
    }

    setAmbientLight (strength) {
        this.ambientLightStrength = strength
    }
    addLight (dir, strength) {
        this.lights.push([
            m4.normalize(dir),
            strength
        ])
    }

    getCurrentBuffer(bufferType) {
        return this.getBufferFromId(bufferType, this.getCurrentBufferId(bufferType))
    }
    getBufferFromId (bufferType, bufferId) {
        if (this.buffers[bufferType].locks[bufferId] === true) {
            return this.buffers[bufferType].buffers[bufferId]
        } else {
            throw new Error(`Buffer ${bufferType} was not locked during get`)
        }
    }
    getCurrentBufferId (bufferType) {
        return this.buffers[bufferType].currentId
    }
    lockBuffer (bufferType) {
        const buffer = this.buffers[bufferType]
        for (let i = 0; i < buffer.buffers.length; ++i) {
            if (buffer.locks[i] === false) {
                buffer.locks[i] = true
                return i
            }
        }
        throw new Error(`Ran out of ${bufferType} buffers`)
    }
    unlockBuffer (bufferType, bufferId) {
        if (this.buffers[bufferType].locks[bufferId] === true) {
            this.buffers[bufferType].locks[bufferId] = false
        } else {
            throw new Error(`Buffer ${bufferType} was not locked during unlock`)
        }
    }
    setCurrentBufferId (bufferType, bufferId) {
        if (this.buffers[bufferType].locks[bufferId]) {
            this.buffers[bufferType].currentId = bufferId
        } else {
            throw new Error(`Buffer ${bufferType} was not locked during set`)
        }
    }

    transformVerts (worldMat, verts) {
        const wvp = m4.multiply(camViewPersp, worldMat)
        const bufferWidth = this.bufferWidth
        const bufferHeight = this.bufferHeight
        const originX = bufferWidth/2 - this.origin[0]
        const originY = bufferHeight/2 - this.origin[1]
        return verts.map(vert => {
            const v = m4.transformVector(wvp, [vert[0], vert[1], vert[2], 1])
            if (v[2] >= -2) {
                const iw = 1/v[3]
                return [v[0]*bufferWidth*iw + originX, v[1]*bufferHeight*iw + originY, -iw]
            } else {
                return null
            }
        })
    }

    drawModel (model, worldMat, colorBuffer, depthBuffer, brightnessBuffer, colorOverride=undefined) {
        const transformedVerts = this.transformVerts(worldMat, model.verts)

        const ambientLightStrength = this.ambientLightStrength
        const lightCount = this.lights.length
        let litNormals
        if (lightCount === 0) {
            litNormals = model.normals.map(normal => ambientLightStrength)
        } else {
            const worldRot = JSON.parse(JSON.stringify(worldMat))
            worldRot[12] = 0
            worldRot[13] = 0
            worldRot[14] = 0
            const cameraFwd = [camViewPersp[2],camViewPersp[6],camViewPersp[10]]
            if (lightCount === 1) {
                const light1Norm = this.lights[0][0]
                const light1Strength = this.lights[0][1]
                litNormals = model.normals.map(normal => {
                    const n = m4.normalize(m4.transformVector(worldRot, [normal[0], normal[1], normal[2], 1]))
                    const dot = n[0]*light1Norm[0] + n[1]*light1Norm[1] + n[2]*light1Norm[2]
                    const brightness = (dot > 0 ? Math.sin(dot) : 0) * light1Strength + ambientLightStrength
                    return brightness
                })
            }
        }

        model.objects.forEach(object => {
        //   if (!excludeObjects.includes(object.name)) {
            object.subObjects.forEach(subObject => {
                const color = colorOverride === undefined ? this.getPaletteId(subObject.diffuse, subObject.emissive) : colorOverride

                let tri = 0
                subObject.tris.forEach(triIndicies => {
                    try {
                        const brightness = litNormals[subObject.triNorms[tri++]]
                        this.clipDrawTri_Cull_Depth(
                            colorBuffer.data,
                            depthBuffer.data,
                            brightnessBuffer.data,
                            transformedVerts[triIndicies[0]],
                            transformedVerts[triIndicies[1]],
                            transformedVerts[triIndicies[2]],
                            color,
                            brightness
                        )
                    } catch (e) {
                        this.renderStats.badFaces++
                    }
                })

                let quad = 0
                subObject.quads.forEach(quadIndicies => {
                    try {
                        const brightness = litNormals[subObject.quadNorms[quad++]]
                        this.clipDrawTri_Cull_Depth(
                            colorBuffer.data,
                            depthBuffer.data,
                            brightnessBuffer.data,
                            transformedVerts[quadIndicies[0]],
                            transformedVerts[quadIndicies[1]],
                            transformedVerts[quadIndicies[2]],
                            color,
                            brightness
                        )
                        this.clipDrawTri_Cull_Depth(
                            colorBuffer.data,
                            depthBuffer.data,
                            brightnessBuffer.data,
                            transformedVerts[quadIndicies[0]],
                            transformedVerts[quadIndicies[2]],
                            transformedVerts[quadIndicies[3]],
                            color,
                            brightness
                        )
                    } catch (e) {
                        this.renderStats.badFaces++
                    }
                })

                let line = 0
                subObject.lines.forEach(lineIndicies => {
                    try {
                        const brightness = 1//litNormals[subObject.quadNorms[quad++]]
                        this.clipDrawLine_Depth(
                            colorBuffer.data,
                            depthBuffer.data,
                            brightnessBuffer.data,
                            transformedVerts[lineIndicies[0]],
                            transformedVerts[lineIndicies[1]],
                            color,
                            brightness
                        )
                    } catch (e) {
                        this.renderStats.badFaces++
                    }
                })
            })
        })
    }

    #edgeFunction(a, b, c) {
        return (c[0] - a[0]) * (b[1] - a[1]) - (c[1] - a[1]) * (b[0] - a[0])
    } 

    clipDrawTri_Cull_Depth(colorBufferData, depthBufferData, brightnessBufferData, v0, v1, v2, color, brightness) {
        this.renderStats.trisConsidered++
        
        if ((v1[0]-v0[0]) * (v1[1]+v0[1]) + 
            (v2[0]-v1[0]) * (v2[1]+v1[1]) +
            (v0[0]-v2[0]) * (v0[1]+v2[1]) >= 0) {
            this.renderStats.trisBackfaceCulled++
            return
        }
    
        const minX = Math.max(Math.floor(Math.min(v0[0], v1[0], v2[0])), this.displayPort[0])
        const minY = Math.max(Math.floor(Math.min(v0[1], v1[1], v2[1])), this.displayPort[1])
        const maxX = Math.min(Math.floor(Math.max(v0[0], v1[0], v2[0])), this.displayPort[2])
        const maxY = Math.min(Math.floor(Math.max(v0[1], v1[1], v2[1])), this.displayPort[3])

        let pixelsVisible = false
        for (let y = minY; y <= maxY; ++y) {
            let bufferIndex = y*this.bufferWidth + minX
            let p = [minX + 0.5, y + 0.5]
            for (let x = minX; x <= maxX; ++x) {
                let w0 = this.#edgeFunction(v1, v2, p)
                if (w0 <= 0) {
                    let w1 = this.#edgeFunction(v2, v0, p)
                    if (w1 <= 0) {
                        let w2 = this.#edgeFunction(v0, v1, p)
                        if (w2 <= 0) {
                            const edge0 = [v2[0]-v1[0], v2[1]-v1[0]]
                            const edge1 = [v0[0]-v2[0], v0[1]-v2[0]]
                            const edge2 = [v1[0]-v0[0], v1[1]-v0[0]]
                            if ((w0 == 0 ? ((edge0[1] == 0 && edge0[0] < 0) || edge0[1] < 0) : (w0 < 0)) &&
                                (w1 == 0 ? ((edge1[1] == 0 && edge1[0] < 0) || edge1[1] < 0) : (w1 < 0)) &&
                                (w2 == 0 ? ((edge2[1] == 0 && edge2[0] < 0) || edge2[1] < 0) : (w2 < 0))) {

                                const area = this.#edgeFunction(v0, v1, v2)
                                w0 /= area
                                w1 /= area
                                w2 /= area
                                const oneOverZ = 1/(v0[2]*w0 + v1[2]*w1 + v2[2]*w2)
                                const z = 1 - oneOverZ
                                if (z < depthBufferData[bufferIndex]) {
                                    colorBufferData[bufferIndex] = color
                                    depthBufferData[bufferIndex] = z
                                    brightnessBufferData[bufferIndex] = brightness
                                    // maxz = Math.max(maxz,z)
                                    // minz = Math.min(minz,z)
                                    // pixelsVisible = true
                                }
                            }
                        }
                    }
                }
                p[0]++
                bufferIndex++
            }
        }

        if (pixelsVisible) {
            this.renderStats.trisRendered++
        } else {
            this.renderStats.trisHidden++
        }
    }

    clipDrawTri_Cull_Depth_Textured(colorBufferData, depthBufferData, brightnessBufferData, v0, v1, v2, tx0, tx1, tx2, texture, color, brightness) {
        this.renderStats.trisConsidered++
        
        if ((v1[0]-v0[0]) * (v1[1]+v0[1]) + 
            (v2[0]-v1[0]) * (v2[1]+v1[1]) +
            (v0[0]-v2[0]) * (v0[1]+v2[1]) >= 0) {
            this.renderStats.trisBackfaceCulled++
            return
        }

        const minX = Math.max(Math.floor(Math.min(v0[0], v1[0], v2[0])), this.displayPort[0])
        const minY = Math.max(Math.floor(Math.min(v0[1], v1[1], v2[1])), this.displayPort[1])
        const maxX = Math.min(Math.floor(Math.max(v0[0], v1[0], v2[0])), this.displayPort[2])
        const maxY = Math.min(Math.floor(Math.max(v0[1], v1[1], v2[1])), this.displayPort[3])

        const textureData = texture.data
        const textureWidth = texture.width-1
        const textureHeight = texture.height-1
        const textureStride = texture.width

        const st0 = [tx0[0]*v0[2], tx0[1]*v0[2]]
        const st1 = [tx1[0]*v1[2], tx1[1]*v1[2]]
        const st2 = [tx2[0]*v2[2], tx2[1]*v2[2]]
        
        const area = this.#edgeFunction(v0, v1, v2)
        let pixelsVisible = false
        for (let y = minY; y <= maxY; ++y) {
            let bufferIndex = y*this.bufferWidth + minX
            let p = [minX + 0.5, y + 0.5]
            for (let x = minX; x <= maxX; ++x) {
                let w0 = this.#edgeFunction(v1, v2, p)
                if (w0 <= 0) {
                    let w1 = this.#edgeFunction(v2, v0, p)
                    if (w1 <= 0) {
                        let w2 = this.#edgeFunction(v0, v1, p)
                        if (w2 <= 0) {
                            const edge0 = [v2[0]-v1[0], v2[1]-v1[0]]
                            const edge1 = [v0[0]-v2[0], v0[1]-v2[0]]
                            const edge2 = [v1[0]-v0[0], v1[1]-v0[0]]
                            if ((w0 == 0 ? ((edge0[1] == 0 && edge0[0] < 0) || edge0[1] < 0) : (w0 < 0)) &&
                                (w1 == 0 ? ((edge1[1] == 0 && edge1[0] < 0) || edge1[1] < 0) : (w1 < 0)) &&
                                (w2 == 0 ? ((edge2[1] == 0 && edge2[0] < 0) || edge2[1] < 0) : (w2 < 0))) {
                                w0 /= area
                                w1 /= area
                                w2 /= area
                                let s = w0 * st0[0] + w1 * st1[0] + w2 * st2[0]
                                let t = w0 * st0[1] + w1 * st1[1] + w2 * st2[1]
                                const oneOverZ = 1/(v0[2]*w0 + v1[2]*w1 + v2[2]*w2)
                                const z = 1 - oneOverZ
                                s *= oneOverZ
                                t *= oneOverZ
                                
                                if (z < depthBufferData[bufferIndex]) {
                                    const b = textureData[(Math.floor(clamp(1-s,0,1)*textureWidth) + Math.floor(clamp(t,0,1)*textureHeight)*textureStride) * 4]
                                    if (b > 0) {
                                        colorBufferData[bufferIndex] = color
                                        depthBufferData[bufferIndex] = z
                                        brightnessBufferData[bufferIndex] = b / 255
                                        // maxz = Math.max(maxz,z)
                                        // minz = Math.min(minz,z)
                                        // pixelsVisible = true
                                    }
                                }
                            }
                        }
                    }
                }
                p[0]++
                bufferIndex++
            }
        }

        if (pixelsVisible) {
            this.renderStats.trisRendered++
        } else {
            this.renderStats.trisHidden++
        }
    }

    // https://github.com/w8r/liang-barsky/
    EPSILON = 1e-6
    clipT(num, denom, c) {
        const [tE, tL] = c
        if (Math.abs(denom) < this.EPSILON) return num < 0
        const t = num / denom
    
        if (denom > 0) {
            if (t > tL) return 0
            if (t > tE) c[0] = t
        } else {
            if (t < tE) return 0
            if (t < tL) c[1] = t
        }
        return 1
    }
    clip(a, b, box, da, db) {
        const [x1, y1] = a
        const [x2, y2] = b
        const dx = x2 - x1
        const dy = y2 - y1

        if (da === undefined || db === undefined) {
            da = a
            db = b
        } else {
            da[0] = a[0]
            da[1] = a[1]
            db[0] = b[0]
            db[1] = b[1]
        }

        if (Math.abs(dx) < this.EPSILON &&
            Math.abs(dy) < this.EPSILON &&
            x1 >= box[0] &&
            x1 <= box[2] &&
            y1 >= box[1] &&
            y1 <= box[3]
            ) {
            return true // this.INSIDE
        }

        const c = [0, 1]
        if (this.clipT(box[0] - x1, dx, c) &&
            this.clipT(x1 - box[2], -dx, c) &&
            this.clipT(box[1] - y1, dy, c) &&
            this.clipT(y1 - box[3], -dy, c)
            ) {
            const [tE, tL] = c
            if (tL < 1) {
                db[0] = x1 + tL * dx
                db[1] = y1 + tL * dy
            }
            if (tE > 0) {
                da[0] += tE * dx
                da[1] += tE * dy
            }
            return true // this.INSIDE
        }
        return false // this.OUTSIDE
    }

    clipDrawLine_Depth (colorBufferData, depthBufferData, brightnessBufferData, p0, p1, color, brightness) {
        let cp0 = []
        let cp1 = []
        if (this.clip(p0, p1, this.displayPort, cp0, cp1)) {
            let x0 = Math.floor(cp0[0])
            let y0 = Math.floor(cp0[1])
            const x1 = Math.floor(cp1[0])
            const y1 = Math.floor(cp1[1])

            const dx = Math.abs(x1 - x0)
            const sx = x0 < x1 ? 1 : -1
            const dy = Math.abs(y1 - y0)
            const sy = y0 < y1 ? 1 : -1 
            let err = (dx > dy ? dx : -dy) / 2

            const z = -1/((p0[2]+p1[2])/2) // ugh..

            while (true) {
                const index = x0 + y0*this.bufferWidth
                if (depthBufferData[index] > z) {
                    colorBufferData[index] = color
                    depthBufferData[index] = z
                    brightnessBufferData[index] = brightness
                }

                if (x0 == x1 && y0 == y1) break
                const e2 = err
                if (e2 > -dx) {
                    err -= dy
                    x0 += sx
                }
                if (e2 < dy) {
                    err += dx
                    y0 += sy
                }
            }
        }
    }
    
    endFrame () {
        this.ctx.putImageData(this.canvasBuffer, -this.overscanSize, -this.overscanSize)
    }

    saveImage () {
        const pixmapSize = this.bufferWidth * this.bufferHeight
        const imageData = new Uint8Array(pixmapSize * 3)
        let writeIndex = 0
        for (let i = 0; i < pixmapSize; ++i) {
            const index = i * 4
            imageData[writeIndex++] = this.canvasBuffer.data[index+0]
            imageData[writeIndex++] = this.canvasBuffer.data[index+1]
            imageData[writeIndex++] = this.canvasBuffer.data[index+2]
        }

        const message = {
            type: 'saveImage',
            data: {
            name: 'name',
                width: this.bufferWidth,
                height: this.bufferHeight,
                imageData
            }
        }
        devserverSend(message)
    }

    getRenderStats () {
        this.renderStats.paletteSize = this.palette.length
        this.renderStats.maxz = maxz
        this.renderStats.minz = minz

        // Janky..
        const x = gui.mouseX + this.overscanSize
        const y = gui.mouseY + this.overscanSize
        if (x >=0 && x < this.bufferWidth && y >=0 && y <this.bufferHeight) {
            this.renderStats.mouseDepth = renderer.getCurrentBuffer('depth').data[x + y*this.bufferWidth]
            this.renderStats.mouseBrightness = renderer.getCurrentBuffer('brightness').data[x + y*this.bufferWidth]
            this.renderStats.mouseColor = renderer.getCurrentBuffer('color').data[x + y*this.bufferWidth]
        } else {
            this.renderStats.mouseDepth = '?'
            this.renderStats.mouseBrightness = '?'
            this.renderStats.mouseColor = '?'
        }

        return this.renderStats
    }
}
let maxz = 0
let minz = 999